
// Shade.C

// This code shades a point lying on a surface which has been intersected by a
// radiance ray. Radiance is aggregated for each of 3 wavelengths (R, G, and B)
// according to the shading model specified.

// system include files
#include <math.h>
#include <stdio.h>
#include <iostream.h>
#include <assert.h>

// local include files
#include "Utils.h"
#include "RTtypes.h"
#include "Interface.h"
#include "RayCast.h"
#include "Shade.h"
#include "Trace.h"


extern int NumHits, NumMiss;       // global hit/miss counts


// ***SHADE***
// Effects: This procedure shades a point lying on a surface which has been intersected by
//  a radiance ray. Radiance is aggregated for each of 3 wavelengths (R, G, B) according to
//  the shading model specified.
//   hit is a data structure containing information about the intersection point to be shaded.
//   view is the incoming radiance ray onto the point
//   depth is the cuurent depth in the recursion tree (used only in recursive shading)
//   weight is this radiance ray's contribution to the radiance of the original ray that
//    spawned it (used only in recursive shading)
//   eta is the index of refraction (>= 1.0) for this ray's travel medium
//   octree is the data structure containing the bounding boxes of all objects in the scene
//   lights is an array of all the lights in the scene
//   nlights is the number of lights in the scene
// Returns: The radiance resulting from shading this point according to the specified shading model:
//  Our shading model is one of :
//  NORMAL:     use normal as color
//  FLAT:       diffuse color only
//  AMBIENT:    ambient light only
//  LOCAL:      AMBIENT + diffuse, specular
//  SHADOWS:    LOCAL with shadow rays
//  RECURSIVE:  spawn reflection, refraction rays
Radiance Shade(Hit &hit,				
               Ray &view,				
               int depth, float weight, float eta,	
               Octree *octree,				
               LightData *lights, int nLights)		

  // BBB -- You need to fill in most of the central "swicth" statement. These are mostly
  // high-level calls to other methods

{
  ObjectData *obj = hit.object;

  // If we didn't hit a object, return background color and quit
  if (!obj)
    {
      NumMiss++;				// update miss counts
      return BackgroundMap(view.dirWS);
    }
  else
    NumHits++;					// update hit counts

  Radiance radiance = Radiance(0, 0, 0);	// initialize radiance to black
  int castshadowrays = FALSE;

  switch (interface::shademode) // cases in brackets to placate the compiler
    {
    case SHADE_NORMAL:
      {
	// use the surface normal as a color
	// map normal [-1..+1] x [-1..+1] x [-1..+1] to radiance
	// (x y z) -> (r g b)

	radiance[0] = 0.5 * (1. + hit.surfnWS[0]);
	radiance[1] = 0.5 * (1. + hit.surfnWS[1]);
	radiance[2] = 0.5 * (1. + hit.surfnWS[2]);
	break;
      }
    case SHADE_FLAT:
      {
	// shade using object's diffuse color

	radiance = obj->diffuse;
	break;
      }
    case SHADE_RECURSIVE:
      {

	// If we haven't gone too deep in our recursion tree, keep recursing to
	//  find reflection and transmission radiance

	if (depth < interface::maxdepth)
	  {
	    // BBB -- Add reflective radiance,
	    //        computed by ReflectionRadiance():
	    radiance += ReflectionRadiance(view, hit.surfpWS, hit.surfnWS,
					   obj, depth, weight, eta,
					   octree, lights, nLights);
	    
	    // BBB - Add transmissive radiance,
	    //       computed by TransmissionRadiance():
	    radiance += TransmissionRadiance(view, hit.entering, hit.surfpWS, hit.surfnWS,
	    obj, depth, weight, eta,
	     octree, lights, nLights);

	    // CCC -- cast other kinds of rays to pick up other visual effects.
	    //        For example, "distribution rays" to handle area lights
	    //        for soft shadows.
	  }

	// note: no break; drop through to collect SHADOWS illumination
      }
    case SHADE_SHADOWS:
      {
	// Here we just set the flag castshadowrays.  In the next section,
	// if this flag is set, we actually cast the shadow rays to determine
	// the degree to which this point is occluded from each light source.
	castshadowrays = TRUE;

	// note: no break; drop through to collect LOCAL illumination
      }
    case SHADE_LOCAL:
      {

	// sum individual contributions of light sources

	for (int k = 0; k < nLights; k++)
	  {
	    LightData *light = &lights[k];
  
	    Vector LdirWS;	// unit vector pointing from shade pt to light
	    float distToLight;	// distance from shade pt to light
	    int Ltype = light->lighttype;


	    // Calculate direction of and/or distance to light source
	    // (depending on its type)

	    if (Ltype == INFINITE_LIGHT)
	      {
		// BBB - compute LdirWS and distToLight: (infinite light)
		//
		//       For an infinite light source, these are trivial.
		//       Think about what it means for a light source to
		//       be infinitely far away.  Does the direction or
		//       distance depend on where the surface point is?
		LdirWS = -(light->directionWS);
		
		distToLight = HUGE;

	      }
	    else         // if LOCAL or SPOT_LIGHT
	      {
		// BBB - compute LdirWS and distToLight: (point light)
		//
		//       For point light sources, the direction and
		//       distance will depend on the surface point.
		//       These are also straightforward.
		LdirWS = light->locationWS - hit.surfpWS;

		distToLight = sqrt(pow((hit.surfpWS[0] - (light->directionWS[0])), 2)
				   + pow((hit.surfpWS[1] - (light->directionWS[1])), 2)
				   + pow((hit.surfpWS[2] - (light->directionWS[2])), 2));
		
	      }

	    //Once you've set LdirWS, normalize it:
	    LdirWS.normalize();
	    
	    float visibility = 1.0;		// percentage of light that gets
                                                // to object

	    // calculate to what degree our point is occluded from the light source

	    if (castshadowrays)
	      {
		// BBB
		//
		// if shadow ray testing is enabled, cast shadow ray
		//        
		//        The actual casting of the shadow ray is done in
		//        the function Shadowing(), which you must write fill in.
		//        Here, just call Shadowing() and set visibility to the
		//        value it returns.
		//
		//        visibility will be 0 in two cases:
		//           the normal is pointing away from the light, or
		//           Shadowing() returns 0
		//
		//        If visibility is zero, there is no need to carry out
		//        the rest of the local illumination calculations!
		visibility = Shadowing(hit.surfpWS, LdirWS, distToLight, octree);
	      }

	    // Now, calculate angular attenuation (for spot lights) and
	    //  distance attenuation (for both spot and point lights)

	    if ( Ltype == SPOT_LIGHT ) {

	      // BBB - If the light is a spotlight compute the 
	      //       angular attenuation.
	      //       Multiply visibility by this value.
	      //
	      //       This is computed by AngularAttenuation(), which 
	      //       you must fill in.
	      visibility *= AngularAttenuation(light, LdirWS);

	    }

	    if (Ltype == LOCAL_LIGHT || Ltype == SPOT_LIGHT) {

	      // BBB - If the light is a spotlight or a local light, 
	      //       compute the dropoff amount
	      //       Multiply visibility by this value.
	      //
	      //       This is computed by DistanceAttenuation(),
	      //       which you must fill in.
	      visibility *= DistanceAttenuation(distToLight);
	    }

    	    // BBB - collect radiance due to diffuse reflection
	    //       with a call to DiffuseRadiance()

	    radiance += DiffuseRadiance(hit, LdirWS, light, visibility);
	    
	    // BBB - collect radiance due to specular reflection
	    //       with a call to SpecularRadiance()
	    
	    radiance += SpecularRadiance(hit, view.dirWS, LdirWS, light, visibility);
	  }

	// note: no break; drop through to collect AMBIENT illumination
      }
    case SHADE_AMBIENT:
      {
	// BBB - collect radiance due to environmental ambient light
	// with a call to AmbiantRadiance() which you must fill in
	radiance += AmbientRadiance(obj);
      
	// BBB - collect radiance due to object's "self-emission":
	// with a call to EmissionRadiance() which you must fill in
	radiance += EmissionRadiance(obj);
	break;
      }
    } // switch

  // clamp RGB output at 1.0
  // CCC -- do something realistic with excess radiance.

  if (radiance[0] > 1.0)
    radiance[0] = 1.0;
  if (radiance[1] > 1.0)
    radiance[1] = 1.0;
  if (radiance[2] > 1.0)
    radiance[2] = 1.0;
  return radiance;

}

// ***BACKGROUNDMAP***
// Effects: This procedure simply returns a radiance value for the background. It is used
//  if no intersection point was found and the radiance ray escaped the scene.
//   D is the WS direction of the incoming ray
// Returns: Radiance
Radiance BackgroundMap(Vector &D)

// CCC -- For extra credit, figure out something more realistic
//        to use here. (For example, a pixel map representing a background
//        or the color of the fog.) For an easy way to do it, let the user
//        provide a color of the command line.

{
  float ssum = sqrtf(D[0]*D[0]+D[1]*D[1]+D[2]*D[2]);			
  float r = fabsf (D[0]/ssum);						
  float g = fabsf (D[1]/ssum);						
  float b = fabsf (D[2]/ssum);						
  return (Radiance(r/4, g/4, b/4));					

}

// ***REFLECTIONDIRECTION***
// Effects: This procedure calculates the direction of the reflected ray
//  when the recursive shading model is used. The reflected ray is returned
//  as rray (WHICH IS A UNIT VECTOR).
//   I is the incident ray
//   N is the surface normal
//   rray the reflected ray direction (AND IT IS A UNIT VECTOR)
// Modifies: rray
void ReflectionDirection(Ray &I, Vector &N, OUTPUT Vector &rray)

// BBB -- Computes the reflection direction of a ray at the interface point.

{
  Vector v = - ((2*N).dot(I.dirWS))*N + I.dirWS;
  v.normalize();
  rray[0] = v[0];
  rray[1] = v[1];
  rray[2] = v[2];

}

// ***TRANSMISSIONDIRECTION***
// Effects: This procedure calculates the direction of the transmission ray
//  when the recursive shading model is used. The transmitted ray is returned
//  as tray.
//   I is the incident ray
//   N is the surface normal
//   etaI is the index of refraction of the material the ray came from
//   etaT is the index of refraction of the material the ray is entering
//   tray the transmitted ray direction
// Modifies: tray
// Returns: 1 iff total internal reflection occurs. 0 otherwise
int TransmissionDirection(Ray &I, Vector &N, float etaI, 
                          float etaT, OUTPUT Vector &tray)
// BBB -- Computes the transmission direction of a ray at the interface point.
  //      You should not use any inverse trig functions.
//	  This function should also be short, 4-8 lines.
{
  Vector t = -(etaI/etaT) * ((-I.dirWS) + (N.dot(I.dirWS))*N) -
    N * sqrt(1 - (etaI*etaI)/(etaT*etaT) * 
             (1 - N.dot(I.dirWS)*N.dot(I.dirWS)));
  
  t.normalize();
  tray[0] = t[0];
  tray[1] = t[1];
  tray[2] = t[2];
  if (N.dot(t) * N.dot(I.dirWS) > 0)
    return 0;
  else
    return 1;
  

}

// ***REFLECTIONRADIANCE***
// Effects: This procedure returns the radiance accumulated by the reflected
//  ray when the recursive shading model is used. In order to do that, we need
//  to spawn a new ray in the reflected direction by calling trace. HERE IS
//  WHERE THE SHADE-TRACE RECURSION HAPPENS.
//   incoming is the incoming ray direction
//   location is the WS point we are shading now
//   surfn is the WS surface normal at this point
//   obj is a pointer to the object we reflected off of
//   depth is the cuurent depth in the recursion tree (used only in recursive shading)
//   weight is this radiance ray's contribution to the radiance of the original ray that
//    spawned it (used only in recursive shading)
//   eta is the index of refraction (>= 1.0) for this ray's travel medium
//   octree is the data structure containing the bounding boxes of all objects in the scene
//   lights is an array of all the lights in the scene
//   nlights is the number of lights in the scene
// Returns: Radiance aggregated in reflected direction
Radiance ReflectionRadiance(Ray incoming,        // where the ray came from
                            Point location,      // where it reflected
                            Vector surfn,        // the normal of the surface
                            ObjectData *obj,     // what it reflected off of
                            int depth,           // the depth of the ray
                            float weight,        // the weight the ray gets
                            float eta,           // stuff you need to pass
                            Octree *octree,      //  to Trace when you call
                            LightData *lights,   //  it recursively.
                            int nLights)

// BBB -- Returns the additional radiance accumulated by the reflected ray.
//        Skip it if it won't matter (i.e., if weighted below
//            interface::minweight)
//        Figure out the direction of the reflected ray
//        Trace the ray, and multiply by the specular reflection  of the 
//        material of the object (Refer Shading Model in Lecture 12).
//        Hints: The weight hasn't been reduced yet. Start the new ray a 
//               short distance from where it really starts to avoid 
//               speckling problems. Remember to increment the depth.
// CCC -- Can distribute the reflected rays to get gloss (Lecture 12)
{
  float ks = obj->ks;
  Vector rraydir;
  Ray reflray;
  Radiance rad;
  // if the weight is above the threshold
  if (ks*weight > interface::minweight) {
    // calculate reflected ray direction (RelflectionDirection() may
    // may prove useful)
    ReflectionDirection(incoming, surfn, rraydir);
    // trace ray and return radiance
    Point temp = location + .001*rraydir;
    reflray.origWS = temp;
    reflray.dirWS = rraydir;
    rad = Trace(reflray, depth+1, weight*ks, eta, octree, lights, nLights);
    rad *= obj->shininess;
    rad[0] *= obj->specular[0];
    rad[1] *= obj->specular[1];
    rad[2] *= obj->specular[2];
    return rad;
  }
  // otherwise, return black
  return Radiance(0, 0, 0);


}

// ***TRANSMISSIONRADIANCE***
// Effects: This procedure returns the radiance accumulated by the transmitted
//  ray when the recursive shading model is used. In order to do that, we need
//  to spawn a new ray in the transmitted direction by calling trace. HERE IS
//  WHERE THE SHADE-TRACE RECURSION HAPPENS.
//   incoming is the incoming ray direction
//   entering is 1 iff we are entering the object here, 0 if leaving
//   location is the WS point we are shading now
//   surfn is the WS surface normal at this point
//   obj is a pointer to the object we reflected off of
//   depth is the cuurent depth in the recursion tree (used only in recursive shading)
//   weight is this radiance ray's contribution to the radiance of the original ray that
//    spawned it (used only in recursive shading)
//   eta is the index of refraction (>= 1.0) for this ray's travel medium
//   octree is the data structure containing the bounding boxes of all objects in the scene
//   lights is an array of all the lights in the scene
//   nlights is the number of lights in the scene
// Returns: Radiance aggregated in transmitted direction
Radiance TransmissionRadiance(Ray incoming,    //parameters same as previous function
			      int entering,
                              Point location,  
                              Vector surfn,     
                              ObjectData *obj,  
                              int depth,        
                              float weight,     
                              float eta,       
                              Octree *octree,   
                              LightData *lights,
                              int nLights)

  // BBB -- Returns the radiance accumulated by the transmitted ray.
  //        Skip it if it won't matter (i.e., if weighted below
  //            interface::minweight)
  //        check if the ray is leaving or entering the object here. Use
  //         TransmissionDirection(incoming, wavelength, surfn, eta_vacuum,
  //                               obj->eta, traydir)
  //         for entering and something similar for leaving. The comments for
  //         RayCast() in RayCast.C explain how to tell whether the ray is
  //         entering the object or not; the way we do it is slightly 
  //         non-standard, so you should read it even if you know what you're
  //         doing.
  //        Hints: The weight hasn't been reduced yet. Start the new ray a 
  //               short distance from where it really starts to avoid 
  //               speckling problems. Remember to increment the depth.
  //	         Be sure to check the return value of TransmissionDirection()!
  // CCC -- Can distribute the transmitted rays to get translucency (Lecture 12)
  
{
  float transp = obj->transparency;
  Ray tray;
  Radiance rad;
  int i;
  // if the weight is above the threshold
  if ((obj->ks)*weight > interface::minweight) {
    Vector traydir;
    //incoming.dirWS = -incoming.dirWS;
    if (entering == 1) // entering the object
      i = TransmissionDirection( incoming, surfn, eta, obj->eta, traydir);
    else if (entering == 0) //leaving
      i = TransmissionDirection(incoming, surfn, obj->eta, 1, traydir);
    if (i == 1)
    return Radiance(0.0, 0.0, 0.0);
    else {
      //trace ray and return radiance
      Point temp = location+.001*traydir;
      tray.dirWS = traydir;
      tray.origWS = temp;
      if (entering == 1)
	rad = Trace(tray, depth+1, weight*transp, obj->eta, octree, lights, nLights);
      else
	rad = Trace(tray, depth+1, weight*transp, 1, octree, lights, nLights);
      // incoming.dirWS = -incoming.dirWS;
      return rad*transp;
    }
  }
  // else, return black
  //incoming.dirWS = -incoming.dirWS;
  return Radiance(0.0, 0.0, 0.0);


}

// ***SHADOWING***
// Effect: Determines visibility of point by repeatedly casting rays
//  towards a light until we are past the light, or we are occluded from
//  the light.
//   start is the WS point we are casting shadow rays from
//   LdirWS is the WS direction we are casting shadow rays in
//   distToLight is the distance from start to the light
//   octree is the data structure containing all bounding boxes of all objects
// Returns: visibility of point by the light (0 is occluded, 1 is perfectly visible)
float Shadowing(Point start, Vector LdirWS, float distToLight, Octree *octree)

// BBB - cast a shadow ray from start to a light distToList away in direction
//       LdirWS. octree contains the scene objects that might get in the way.
//       Remember to displace away from surfaces to avoid speckling. Return how
//       much of the light will get through to the object (i.e. 0.0 - 1.0).
//       Only count a ray as blocked if it is entering an object.  (Ignore
//       rays leaving an object.)

{
  Hit hitobject;
  float trans = 1.0;

  // displace slightly away from the object surface
  // to avoid the problem of speckling in your output.

  Point tstart = start+.001*LdirWS; 
  Ray light_ray;
  light_ray.dirWS = LdirWS;
  light_ray.origWS = tstart;
  Point newstart = tstart;
//  do 		// accumulate medium transparency
//  while (there is still an object between us at the light
//         && the light is not completely blocked (trans>0)
//         && we are not past the light source);

  while ((RayCast(light_ray, octree, hitobject) != 0) &&
	   ((trans > 0) && ((newstart - tstart).length() < distToLight))) { 
      if(hitobject.entering) {
	trans *= hitobject.object->transparency;
      }
      newstart = hitobject.surfpWS + .001*LdirWS;
      light_ray.origWS = newstart;
    }
    return trans;
  
}

// ***ANGULARATTENUATION***
// Effects: Calculate how much of the light (shining in its original direction)
//  is still visible at the angle the light is at (relative to the point).
//  This is only used for spotlights.
//   light is a pointer to the light data
//   LdirWS is the WS direction TO the light source (different from
//    the direction the light is shining FROM)
// Returns: fraction of light inpinging on point

float AngularAttenuation(LightData *light, Vector LdirWS)

// BBB -- return how much of the light is still at the angle the light is at.
//        take into account the cutoff angle.
//
//        Hint: some useful values:
//        		light->directionWS
//        		spotlight->cutOffAngle.getValue()
//        		spotlight->dropOffRate.getValue()

{
  SoSpotLight *spotlight = (SoSpotLight *) (light->light);
  
  float cosine = -LdirWS.dot(light->directionWS) /
    ((LdirWS.length() * (light->directionWS).length()));
  float angle = acos(cosine);
  if (angle > spotlight->cutOffAngle.getValue()) {
    return 0.0;
  } else {
    float result = 1 - powf(angle / spotlight->cutOffAngle.getValue(), 4.0);
    
    
    if (result > 1.0) return 1.0;
    if (result < 0.0) return 0.0;
    return result;
  }
}

// ***DISTANCEATTENUATION***
// Effects: Calculate how much of the light is still visible at the distance the point
//  is at from the light.
//  This is only used for spotlights and local (point) lights.
//   distToLight is the distance from the point to the light source.
// Returns: fraction of light inpinging on point
  float DistanceAttenuation(float distToLight)
    {
  // BBB -- return how much the light is left at the distance from the source.
  //
  // We use the PHONG distance attenuation function, which is of the form
  //
  //  fatt = min(1, 1/(c1 + c2*d + c3*(d^2)))
  //
  //   where d is the distance to the light source
  //   and   c1, c2, and c3 are constants between 0 and 1
  // Here, we arbitrarily set c1, c2, and c3 to 0, 1, and 0 respectively

  
    float c1 = 0.0, c2 = 1.0, c3 = 0.0;
    float d = distToLight;
    float fatt = 1/(c1 + c2*d + c3*d*d);    
    if (1.0 < fatt)
      return 1.0;
    else
      return fatt;


}


// ***DIFFUSERADIANCE***
// Effects: This procedure collects the diffuse radiance emanating from the surface
//  at the point we are shading.
//   hit is the data structure containing information about the point we are shading
//   LdirWS is the WS direction to the light source
//   light is a pointer to the light data
//   V is the fraction of emitted light from the source actually reaching the point
// Returns: Diffuse radiance from this point
Radiance DiffuseRadiance(Hit hit, Vector LdirWS, LightData *light, float V)

// BBB -- collect diffuse radiance
// Hint: light->light-><variable>.getValue() will get you light info.
//       material properties come from hit.object-><variable>

{
  
float n_dot_l = LdirWS.dot(hit.surfnWS);

  if (n_dot_l < 0) n_dot_l = 0;

    Color lightColor = light->light->color.getValue();

    Color newcolor(lightColor[0] * hit.object->diffuse[0],
		   lightColor[1] * hit.object->diffuse[1],
		   lightColor[2] * hit.object->diffuse[2]);

  return V*light->light->intensity.getValue() * newcolor*(n_dot_l);
}

// ***SPECULARRADIANCE***
// Effects: This procedure collects the specular radiance emanating from the surface
//  at the point we are shading.
//   hit is the data structure containing information about the point we are shading
//   viewdWS is the WS direction in which the incoming ray points
//   LdirWS is the WS direction to the light source
//   light is a pointer to the light data
//   V is the fraction of emitted light from the source actually reaching the point
// Returns: Specular radiance from this point
Radiance SpecularRadiance(Hit hit, Vector viewdWS, Vector LdirWS,
                           LightData *light, float V)

// BBB -- collect specular radiance 
// See DiffuseRadiance for hints.

{
  Color obj_color = hit.object->specular;
  Vector h = (LdirWS - viewdWS)/((LdirWS - viewdWS).length());
  float shiny = hit.object->shininess;
  float ndoth = hit.surfnWS.dot(h);
  if (ndoth < 0)
    ndoth = 0;
  Radiance rad = V*(light->light->intensity.getValue())*obj_color*(pow(ndoth, 256*shiny));
  return rad;

}

// ***AMBIENTRADIANCE***
// Effects: This procedure collects the ambient radiance emanating from the surface
//  at the point we are shading.
//   obj is a pointer to the object we are shading.
// Returns: Ambient radiance from this point
Radiance AmbientRadiance(ObjectData *obj)

// BBB -- collect radiance due to environmental ambient light
// Hint: obj->environment-><variable>.getValue() return environmental info.

{ 

 Radiance env_radiance = obj->environment->ambientColor.getValue();
  Radiance obj_radiance = obj->ambient;
  float env_intensity = (obj->environment->ambientIntensity.getValue());
  env_radiance[0] *= obj_radiance[0];
  env_radiance[1] *= obj_radiance[1];
  env_radiance[2] *= obj_radiance[2];
  return env_radiance * env_intensity;


}

// ***EMISSIONRADIANCE***
// Effects: This procedure collects the emissive radiance emanating from the surface
//  at the point we are shading.
//   obj is a pointer to the object we are shading.
// Returns: Emissive radiance from this point
Radiance EmissionRadiance(ObjectData *obj)

// BBB -- return object's own radiance

{
  return obj->emission;
}



