import Raster;

public class Matrix3D {
  float m[][] = new float[4][4];
  float raster_w = 0.0f, raster_h = 0.0f;

  //
  // Constructors
  //
  public Matrix3D() {
    // initialize with identity transform
    for(int i = 0; i < 4; i ++)
      for(int j = 0; j < 4; j ++) {
	if(i == j)
	  m[i][j] = 1.0f;
	else
	  m[i][j] = 0.0f;
      }
  }

  public Matrix3D(Matrix3D copy) {
    // initialize with copy of source
    for(int i = 0; i < 4; i ++)
      for(int j = 0; j < 4; j ++)
	m[i][j] = copy.get(i,j);
  }
    
  public Matrix3D(Raster r) {
    // initialize with a mapping from canonical space to screen space
    loadIdentity();
    raster_w = (float)r.getWidth();
    raster_h = (float)r.getHeight();
  }
    
  //
  // General interface methods
  //        
  public void set(int i, int j, float value) {
    // set element [i][j] to value
    m[i][j] = value;
  }
  public float get(int i, int j) {
    // return element [i][j]
    return m[i][j];
  }
  
  //
  // Transform points from the in array to the out array
  // using the current matrix. The subset of points transformed
  // begins at the start index and has the specified length
  //
  //    for (i = 0; i < length; i++)
  //        out[start+i] = this * in[start+i]
  //
  public void transform(Point3D in[], Point3D out[], int start, int length) {

    for( int i = 0; i < length; i++) {
      float h = (m[3][0] * in[start+i].x +
		 m[3][1] * in[start+i].y +
		 m[3][2] * in[start+i].z +
		 m[3][3]);
      if(h != 0.0f ) {
	out[start+i].x = (m[0][0] * in[start+i].x +
			  m[0][1] * in[start+i].y +
			  m[0][2] * in[start+i].z +
			  m[0][3]) / h;
	
	out[start+i].y = (m[1][0] * in[start+i].x +
			  m[1][1] * in[start+i].y +
			  m[1][2] * in[start+i].z +
			  m[1][3]) / h;

	out[start+i].z = (m[2][0] * in[start+i].x +
			  m[2][1] * in[start+i].y +
			  m[2][2] * in[start+i].z +
			  m[2][3]) / h;
      }
      else
	System.out.println("Infinite points!");
    }
  }	
  
  public final void compose(Matrix3D src) {
    // this = this * src
    float temp[] = new float[4];
    
    for(int i = 0; i < 4; i++) {
      temp[0] = m[i][0];
      temp[1] = m[i][1];
      temp[2] = m[i][2];
      temp[3] = m[i][3];
      for(int j = 0; j < 4; j++) {
	float tempp = 0.0f;
	for(int k = 0; k < 4; k++) 
	  tempp = tempp + temp[k] * src.get(k,j);
	m[i][j] = tempp;
      }
    }
    for(int i = 0; i < 4; i++)
      for(int j = 0; j < 4; j++) 
	System.out.println("src["+i+"]["+j+"]="+src.get(i,j));
  }
    
  public void loadIdentity() {
    // this = identity
    for(int i = 0; i < 4; i ++)
      for(int j = 0; j < 4; j ++) {
	if(i == j)
	  m[i][j] = 1.0f;
	else
	  m[i][j] = 0.0f;
      }
  }
  
  public void translate(float tx, float ty, float tz) {
    // this = this * t
    Matrix3D trans = new Matrix3D();

    trans.set(0, 3, tx);
    trans.set(1, 3, ty);
    trans.set(2, 3, tz);
    compose(trans);
  }

  public void scale(float sx, float sy, float sz) {
    // this = this * scale
    Matrix3D scal = new Matrix3D();
    
    scal.set(0, 0, sx);
    scal.set(1, 1, sy);
    scal.set(2, 2, sz);
    compose(scal);
  }
    
  public void shear(float kxy, float kxz, float kyz) {
    // this = this * shear
    Matrix3D shea = new Matrix3D();

    shea.set(0, 1, kxy);
    shea.set(0, 2, kxz);
    shea.set(1, 2, kyz);
    compose(shea);
  }

  public void skew(float ax, float ay, float az) {
    // this = this * skew
    Matrix3D skeww = new Matrix3D();

    skeww.set(0, 0, 0.0f);
    skeww.set(1, 1, 0.0f);
    skeww.set(2, 2, 0.0f);
    skeww.set(3, 3, 0.0f);
    skeww.set(0, 1, -az);
    skeww.set(0, 2, ay);
    skeww.set(1, 0, az);
    skeww.set(1, 2, -ax);
    skeww.set(2, 0, -ay);
    skeww.set(2, 1, ax);
    compose(skeww);
  }

  public void rotate(float ax, float ay, float az, float angle) {
    // this = this * rotate
    Matrix3D rot = new Matrix3D();

    float v = (float)Math.sqrt((double) (ax*ax + ay*ay + az*az));
    ax = ax / v;
    ay = ay / v;
    az = az / v;

    float weight1 = 1.0f - (float) Math.cos(angle);
    float weight2 = (float) Math.sin(angle);
    float weight3 = (float) Math.cos(angle);
    rot.set(0, 0, ax*ax*weight1 + weight3);
    rot.set(1, 1, ay*ay*weight1 + weight3);
    rot.set(2, 2, az*az*weight1 + weight3);
    rot.set(0, 1, ax*ay*weight1 - az*weight2);
    rot.set(0, 2, ax*az*weight1 + ay*weight2);
    rot.set(1, 0, ax*ay*weight1 + az*weight2);
    rot.set(1, 2, ay*az*weight1 - ax*weight2);
    rot.set(2, 0, ax*az*weight1 - ay*weight2);
    rot.set(2, 1, ay*az*weight1 + ax*weight2);
    compose(rot);
  }
      
  public void lookAt(float eyex, float eyey, float eyez,
		     float atx,  float aty,  float atz,
		     float upx,  float upy,  float upz) {
    // this = this * lookat
    Matrix3D lAt = new Matrix3D();
    
    float lx = atx - eyex;
    float ly = aty - eyey;
    float lz = atz - eyez;
    float v = (float)Math.sqrt((double) (lx*lx + ly*ly + lz*lz));
    lx = lx / v;
    ly = ly / v;
    lz = lz / v;

    float rx = ly*upz - lz*upy;
    float ry = lz*upx - lx*upz;
    float rz = lx*upy - ly*upx;
    v = (float)Math.sqrt((double) (rx*rx + ry*ry + rz*rz));
    rx = rx / v;
    ry = ry / v;
    rz = rz / v;

    float ux = ry*lz - rz*ly;
    float uy = rz*lx - rx*lz;
    float uz = rx*ly - ry*lx;
    v = (float)Math.sqrt((double) (ux*ux + uy*uy + uz*uz));
    ux = ux / v;
    uy = uy / v;
    uz = uz / v;

    lAt.set(0, 0, rx);
    lAt.set(0, 1, ry);
    lAt.set(0, 2, rz);
    lAt.set(0, 3, -rx*eyex - ry*eyey - rz*eyez);
    lAt.set(1, 0, ux);
    lAt.set(1, 1, uy);
    lAt.set(1, 2, uz);
    lAt.set(1, 3, -ux*eyex - uy*eyey - uz*eyez);
    lAt.set(2, 0, -lx);
    lAt.set(2, 1, -ly);
    lAt.set(2, 2, -lz);
    lAt.set(2, 3, lx*eyex + ly*eyey + lz*eyez);
    compose(lAt);
  }    
    
  //
  // Assume the following projection transformations
  // transform points into the canonical viewing space
  //

  public void perspective(float left, float right,   // this = this * persp
			  float bottom, float top,
			  float near, float far) {
    // this = this * persp
    Matrix3D persp = new Matrix3D();

    persp.set(0, 0, raster_w*near/(right - left));
    persp.set(0, 2, -left*raster_w/(right - left));
    persp.set(1, 1, raster_h*near/(bottom - top));
    persp.set(1, 2, -raster_h*top/(bottom - top));
    persp.set(3, 2, 1.0f);
    persp.set(3, 3, 0.0f);
    
    compose(persp);
  }

  public void orthographic(float left, float right,
			   float bottom, float top,
			   float near, float far) {
    // this = this * ortho
    Matrix3D orth = new Matrix3D();
    
    orth.set(0, 0, raster_w/(right - left));
    orth.set(0, 3, -left*raster_w/(right - left));
    orth.set(1, 1, raster_h/(bottom - top));
    orth.set(1, 3, -top*raster_h/(bottom - top));
    compose(orth);
  }
}






