public class Matrix3D {
  //
  // Constructors
  //
  private float[][] matrix;
  //*****************************************************************
  public Matrix3D()      
       // initialize with identity transform
  {
    matrix = new float[4][4];
    for (int x=0; x<4; x++)
      for (int y=0; y<4; y++)
	if (x==y)
	  matrix[x][y] = 1;
	else
	  matrix[x][y] = 0;
  }
  //*****************************************************************
  public Matrix3D(Matrix3D copy)
       // initialize with copy of source
  {
    matrix = new float[4][4];
    for (int x=0;x<4;x++)
      for (int y=0;y<4;y++)
	matrix[x][y] = copy.get(y,x);
  }
  //*****************************************************************
  public Matrix3D(float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p)
  {
    matrix = new float[4][4];
    matrix[0][0] = a;
    matrix[0][1] = b;
    matrix[0][2] = c;
    matrix[0][3] = d;
    matrix[1][0] = e;
    matrix[1][1] = f;
    matrix[1][2] = g;
    matrix[1][3] = h;
    matrix[2][0] = i;
    matrix[2][1] = j;
    matrix[2][2] = k;
    matrix[2][3] = l;
    matrix[3][0] = m;
    matrix[3][1] = n;
    matrix[3][2] = o;
    matrix[3][3] = p;
  }
  //*****************************************************************
  public Matrix3D(Raster r)          
       // initialize with a mapping from
       // canonical space to screen space
       // General interface methods 
  {
    float width = r.getWidth();
    float height = r.getHeight();
    matrix = new float[4][4];
    for (int x=0;x<4;x++)
      for (int y=0;y<4;y++)
	matrix[x][y]=0;
    matrix[0][0] = width/2;
    matrix[1][1] = height/2;
    matrix[0][3] = width/2;
    matrix[1][3] = height/2;
    matrix[3][3] = 1;
    matrix[2][2] = 1/2;
    matrix[2][3] = -1/2;
  }
  //*****************************************************************
  public void set(int i, int j, float value)
       // set element [j][i] to value
  {
    matrix[j][i] = value;
  }
  //*****************************************************************
  public void showMatrix()
  {
    System.out.println();
    for (int i=0;i<4;i++)
      {
	for (int j=0;j<4;j++)
	  System.out.print(matrix[i][j] + " ");
	System.out.println();
      }
    
  }
  
  //*****************************************************************
  public float get(int i, int j)                      
       // return element [j][i]
  {
    return matrix[j][i];
  }
  //*****************************************************************
  private Point3D matrixPointMultiply(Point3D p3d)
  {
    Point3D temp;
    float x, y, z, l;
    float valx, valy, valz;
    valx = p3d.x;
    valy = p3d.y;
    valz = p3d.z;
    x = (matrix[0][0] * valx) + (matrix[0][1] * valy) + (matrix[0][2] * valz) + matrix[0][3];
    y = (matrix[1][0] * valx) + (matrix[1][1] * valy) + (matrix[1][2] * valz) + matrix[1][3];
    z = (matrix[2][0] * valx) + (matrix[2][1] * valy) + (matrix[2][2] * valz) + matrix[2][3];
    l = (matrix[3][0] * valx) + (matrix[3][1] * valy) + (matrix[3][2] * valz) + matrix[3][3];
    x = x/l;
    y = y/l;
    z = z/l;
    
    temp = new Point3D(x, y, z);
    return temp;
  }
  //*****************************************************************
  public void transform(Point3D in[], Point3D out[], int start, int length)
  {
    
    //  Transform points from the in array to the out array
    //  using the current matrix. The subset of points transformed
    //  begins at the start index and has the specified length
    //  for (i = 0; i < length; i++)
    //  out[start+i] = this * in[start+i]
    //  
    for (int i=0; i<length; i++)
      out[start+i] = this.matrixPointMultiply(in[start+i]);
  }
  //*****************************************************************
  public final void compose(Matrix3D src)
       // this = this * src
  {
    float temp[][];
    temp = new float[4][4];
    float sum = 0;
    for (int row=0; row < 4; row++)
      for (int col=0; col < 4; col++)
	{
	  for (int x=0; x < 4; x++)
	    {
	      sum = sum + (matrix[row][x] * src.get(col,x));
	    }
	  temp[row][col] = sum;
	  sum = 0;
	}
    for (int i=0; i<4; i++)
      for (int j=0; j<4; j++)
	matrix[i][j] = temp[i][j];
  }
  //*****************************************************************
  public void loadIdentity()
       // this = identity
  {
    for (int x=0; x<4; x++)
      for (int y=0; y<4; y++)
	if (x==y)
	  matrix[x][y] = 1;
	else
	  matrix[x][y] = 0;
  }
  //*****************************************************************
  public void translate(float tx, float ty, float tz)
       // this = this * t
  {
    Matrix3D trans;
    trans = new Matrix3D();
    trans.set(3,0,tx);
    trans.set(3,1,ty);
    trans.set(3,2,tz);
    compose(trans);
  }
  //*****************************************************************
  

  public void scale(float sx, float sy, float sz) 
       // this = this * scale
  {
    Matrix3D sc;
    sc = new Matrix3D();
    sc.set(0,0,sx);
    sc.set(1,1,sy);
    sc.set(2,2,sz);
    compose(sc);
  }
  //*****************************************************************
  public void shear(float kxy, float kxz, float kyz)
       // this = this * skew
  {
    Matrix3D sk;
    sk = new Matrix3D();
    sk.set(1,0,kxy);
    sk.set(2,0,kxz);
    sk.set(2,1,kyz);
    compose(sk);
  }
  //*****************************************************************
  public void rotate(float ax, float ay, float az, float angle)
       // this = this * rotate 
  {
    Matrix3D rot;
    rot = new Matrix3D();
    
    // Normalize vector
    float length = (float)Math.sqrt((ax*ax)+(ay*ay)+(az*az));
    ax = ax / length;
    ay = ay / length;
    az = az / length;

    float cost = (float)Math.cos(angle);
    float sint = (float)Math.sin(angle);
    float f1 = 1 - cost;
    
    //Using the formula found in the lecture notes,
    //find the appropriate components of the matrix
    float a = ((ax * ax) * f1) + cost;
    float b = (ax * ay * f1) - (az * sint);
    float c = (ax * az * f1) + (ay *sint);
    float d = (ax * ay * f1) + (az *sint);
    float e = (ay * ay * f1) + cost;
    float f = (ay * az * f1) -(ax *sint);
    float g = (ax * az * f1) - (ay * sint);
    float h = (ay * az * f1) + (ax * sint);
    float i = (az * az * f1) + cost;

    //set the values
    rot.set(0,0,a);
    rot.set(1,0,b);
    rot.set(2,0,c);
    rot.set(3,0,0);
    rot.set(0,1,d);
    rot.set(1,1,e);
    rot.set(2,1,f);
    rot.set(3,1,0);
    rot.set(0,2,g);
    rot.set(1,2,h);
    rot.set(2,2,i);
    rot.set(3,3,1);
    
    compose(rot);
  }
  //*****************************************************************
  public void lookAt(float eyex, float eyey, float eyez,
		     float atx,  float aty,  float atz,
		     float upx,  float upy,  float upz)
       // this = this * lookat
  {
    //This procedure was written with the help of 
    // lecture notes 12
    float lx = atx - eyex;
    float ly = aty - eyey;
    float lz = atz - eyez;
    float llength = (float)Math.sqrt((lx*lx) + (ly*ly) + (lz * lz));
    //normalize l
    float nlx = lx / llength;
    float nly = ly / llength;
    float nlz = lz / llength;
    float rx = -(lz * upy) + (ly * upz);
    float ry = (lz * upx) - (lx * upz);
    float rz = -(ly*upx) + (lx * upy);
    float rlength = (float)Math.sqrt((rx*rx) + (ry*ry) + (rz*rz));
    //normalize r
    float nrx = rx / rlength;
    float nry = ry / rlength;
    float nrz = rz / rlength;
    
    float ux = -(rz * ly) + (ry * lz);
    float uy = (rz * lx) - (rx * lz);
    float uz = -(ry * lx) + (rx * ly);
    
    float ulength = (float)Math.sqrt((ux*ux)+(uy*uy)+(uz*uz));
    //normalize u
    float nux = ux / ulength;
    float nuy = uy / ulength;
    float nuz = uz / ulength;

    Matrix3D look;
    look = new Matrix3D(nrx, nry, nrz, -((nrx*eyex)+(nry*eyey)+(nrz*eyez)), nux, nuy, nuz, -((nux*eyex)+(nuy*eyey)+(nuz*eyez)), -nlx, -nly, -nlz, ((nlx*eyex)+(nly*eyey)+(nlz*eyez)), 0, 0, 0, 1);
    
    compose(look);
  }
  //
  // Assume the following projection transformations
  // transform points into the canonical viewing space
  //
  public void perspective(float left, float right,   // this = this * persp
			  float bottom, float top,
			  float near, float far)
  {
    // This procedure was written with the help of lecture notes 12
    Matrix3D per = new Matrix3D();
    per.set(0,0,((2*near)/(right-left)));
    per.set(2,0,-((right+left)/(right-left)));
    per.set(1,1,((2*near)/(bottom-top)));
    per.set(2,1,-((bottom+top)/(bottom-top)));
    per.set(2,2,((far+near)/(far-near)));
    per.set(3,2,-((2*far*near)/(far-near)));
    per.set(2,3,1);
    per.set(3,3,0);
    compose(per);
  }
  //*****************************************************************
  public void orthographic(float left, float right,  // this = this * ortho
			   float bottom, float top,
			   float near, float far)
  {
    // This procedure was written with the help of lecture notes 12
    Matrix3D ortho = new Matrix3D();
    ortho.set(0,0,(2/(right-left)));
    ortho.set(3,0,-((right+left)/(right-left)));
    ortho.set(1,1,(2/(bottom-top)));
    ortho.set(3,1,-((bottom+top)/(bottom-top)));
    ortho.set(2,2,(2/(far-near)));
    ortho.set(3,2,-((far+near)/(far-near)));
    compose(ortho);
  }
}


