/*
 * rcsched.c
 * 
 * Scheduler for Rational Clocking, integer version
 *    by Luis F. G. Sarmenta
 *
 *    thesis version, 950525
 *
 * Notes: 1) Here, coincidence cycles are called "beat" cycles, as they
 *           are in Pratt and Ward's original paper.
 *
 *        2) All timing parameters are treated internally as offsets 
 *           in the positive time direction from the clock edge they belong 
 *           to.  Thus, setup times are internally negative.  
 *           However, the user can still enter _positive_ setup times as 
 *           these get converted automatically.
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <math.h>


/* standard min and max definitions */

#define min(a,b)   ((a) < (b) ? (a) : (b))
#define max(a,b)   ((a) > (b) ? (a) : (b))


/*
 * modulo and div operators modified to work correctly
 *    with negative numbers
 *
 * This code was developed on a Sun (mont-blanc.lcs.mit.edu) with gcc.
 *    Under this system type, the / and % operators work as follows:
 *        22 / 7 =  3;   22 % 7 =  1;   22 / -7 = -3;   22 % -7 =  1;
 *       -22 / 7 = -3;  -22 % 7 = -1;  -22 / -7 =  3;  -22 % -7 = -1;
 *
 *    The way it handles the mod of negative numbers allows us to
 *    define a correct mod operator as follows:
 */

long mod( long a, long b )
{
   long temp;

   temp = a % b;
   if ( temp < 0 )
      temp += b;
   return( temp );
}

long ndiv( long a, long b )                 /*  ndiv(a,b)*b + mod(a,b) = a  */
{
   return( (a < 0) ? ((a-(b-1))/b) : (a/b) );
}

#define ndivup(a,b) ((ndiv((a)+((b)-1), (b))))        /* ndiv then round-up */

long gcd( long a, long b )
{
   if (!b)
      return( a );
   else
      return( gcd( b, a % b ) );
}


/*
 * Some Global Constants
 */


#define MAXMN   64               /* Max. value for M and N. */
#define ROM_B    4               /* bits for specifying M and N to ROM */
#define ROM_N   (1<<ROM_B)       /* equivalent max M and N for ROM */


long d  = 1000;                  /* smallest unit of time = (delta T) / d  */
long Tref;                       /* reference clock's period in picoseconds.  
                                    This will be initialized properly later. */

/* Options Variables 
 */

int PEEK = 0;                   /* Show intermediate results? */

int DRAWSCHEDS = 1;             /* Draw text schedule diagrams? */

#define DTUNITS 1
#define DLUNITS 2
#define FRUNITS 3
#define CDIVPS  4
#define CMULPS  5

int paramunits = DLUNITS;       /* Parameter Units */

#define PAIR 1
#define TABLE 2

int compmode = PAIR;            /* Compute Mode */

#define NONE 0
#define ORIG 1
#define GBUFF2 2
#define DBUFFG 3
#define DBUFFL 4
#define MAXALGO 4

int algorithm = 1;              /* Scheduling Algorithm */

#define TEXT  1 
#define LDIAG 2
#define LPLOT 3
#define ROM   4

int outmode = LDIAG;            /* Output format */

int usewithchip = 0;            /* Use with prototype chip */

char outfilename[41] = "rcsched.log";       /* Log file name */

FILE *outfile;


/*
 * Primitive operations ('P' stands for primitive)
 * 
 * These operations assume that M and N are integers representing 
 * the normalized frequencies.
 *
 * Most of these operations work in units of dt = (delta t) / d.
 */

   /* The following macros are self-explanatory. */

#define Pbeatperiod(M,N)      ( (M)*(N)*d )
#define Pperiod(M,N)          ( (N)*d )
#define Pfreq(M,N)            ( 1 / Pperiod((M),(N)) )
#define Pdeltat(M,N)          d


   /* Pclockedge gets ith clock edge of system M  */

#define Pclockedge(M,N,i)     ( (i)*(N)*d )

   /* Pcycle returns the cycle which time t belongs to 
    * Pupcycle returns the next cycle at or after time t 
    * note that if t is exactly at some edge, then Pcycle and Pupcycle
    * return the same cycle. 
    */

#define Pcycle(M,N,t)         (ndiv( (t), (d*(N)) ))
#define Pupcycle(M,N,t)       (ndivup( (t), (d*(N)) ))

   /* Pshcycle gets the current cycle if t is a timing param edge that is dt 
    * units away from its clock edge.
    */

#define Pshcycle(M,N,t,dt)    (Pcycle((M),(N),(t)-(dt)))

   /* Pfracttodt converts from a timing param in the fractional form num/d 
    * to its equivalent value in units of dt.
    * (Move this somewhere else?)
    */

#define Pfractodt(M,N,num)    ((num) * ( Pperiod((M),(N)) / d ) )
#define Pdelttodt(M,N,num)    ((num) * ( Pdeltat((M),(N)) ))

   /* The following macros convert from a timing param in units of
    * picoseconds to its equivalent in units of dt, using the reference
    * clock period, Tref.  Pmpstodt is used for clock multiplication,
    * where Tref is the clock period of M.  Pdpstodt is used for
    * clock division, where Tref is Thigh.
    *
    ******
    * Note that for now, these macros round _down_ to the nearest integer.
    * If you want to be conservative, you should really round _down_
    * if the value is contamination delay, but round _up_ otherwise.
    * Possibly fix this later.  For now, just pick d and Tref such that
    * the result is an integer anyway.
    ******
    */

#define Pmpstodt(M,N,num)     (((num) * (N) * d ) / Tref )
#define Pdpstodt(M,N,num)     (((num) * d) / Tref )

   /* Ptptodt converts the value tp to units of dt according to the
    * current units used for specifying tp.
    *
    * This allows the units of tp to change without changing the macros.
    * It also has the nice property that the appropriate scaling rule 
    * for different M and N is naturally taken care of.
    *
    * Note that in these macros, parameters called 'dt' are in units
    * of dt, while parameters called 'tp' are in whatever units the scaling
    * rule specifies.
    */

long Ptptodt( long M, long N, long tp )
{
   switch (paramunits) {
      case DTUNITS : return( tp );
      case FRUNITS : return( Pfractodt( M, N, tp ) );
      case DLUNITS : return( Pdelttodt( M, N, tp ) );
      case CMULPS : return( Pmpstodt( M, N, tp ) );
      case CDIVPS : return( Pdpstodt( M, N, tp ) );
      default: { 
                  fprintf(stderr,"Invalid Param Units ...  using dt ...\n" );
                  return( tp );
	       }
   }
}


   /* Now we use Ptptodt in Pshedge, which gives timing edge i if 
    * timing param is tp away from edge. 
    */

#define Pshedge(M,N,i,tp)     ( Pclockedge((M),(N),(i)) + Ptptodt(M,N,tp) )

   /*
    * The following macros return the appropriate cycle of the _other_ clock.
    * (Note the switch in M and N.)
    */

#define PnxtOcycle(M,N,t)        Pupcycle( (N), (M), (t) )
#define PprvOcycle(M,N,t)        Pcycle( (N), (M), (t) )

#define PnxtOshcycle(M,N,t,dt)   Pupcycle( (N), (M), (t)-(dt) )
#define PprvOshcycle(M,N,t,dt)   Pshcycle( (N), (M), (t), (dt) )


/*
 * Data Structures
 *
 */

typedef int  sched[MAXMN];    /* Schedule table */

typedef struct systype {
   long f;              /* frequency = cycles per beat */
   long S,H,C,P;        /* timing params in units of dt = (delta t) / d */
   sched TE, Tsel;      /* transmit register control signals */
   sched TD;            /* input to transmit register */
   sched Tto;           /* cycle to transmit to (mod beat period) */
   sched RE, Rsel;      /* receive register control signals */
   sched RQ;            /* output of receive register */
   sched Rfrom;         /* cycle to receive from (mod beat period) */
   sched cnt;           /* counter value */
   sched checked;       /* Flag to tell if cycle has already been tried. */
   
   int   perf;          /* Performance as a transmitter. */  

   struct systype *other;       /* pointer to the other system  */
}  systype;


/* These macros are the same as the primitive macros but
 * work with systype structs instead.  Note the capitalization of the names.
 */

#define Mf(M)               (M->f)
#define Nf(M)               ((M->other)->f)

#define Beatperiod(M)       Pbeatperiod( Mf(M), Nf(M) )
#define Period(M)           Pperiod( Mf(M), Nf(M) )
#define Freq(M)             Pfreq( Mf(M), Nf(M) )
#define DeltaT(M)           PdeltaT( Mf(M), Nf(M) )

#define Clockedge(M,i)      Pclockedge( Mf(M), Nf(M), (i) )
#define Cycle(M,t)          Pcycle( Mf(M), Nf(M), (t) )
#define SHcycle(M,t,dt)     Pshcycle( Mf(M), Nf(M), (t), (dt) )

#define Fractodt(M,num)     Pfractodt( Mf(M), Nf(M), (num) )
#define Delttodt(M,num)     Pdelttodt( Mf(M), Nf(M), (num) )

#define Tptodt(M,tp)        Ptptodt( Mf(M), Nf(M), (tp) )
#define SHedge(M,i,tp)      Pshedge( Mf(M), Nf(M), (i), (tp) )

#define NxtOcycle(M,t)      PnxtOcycle( Mf(M), Nf(M), (t) )
#define PrvOcycle(M,t)      PprvOcycle( Mf(M), Nf(M), (t) )
#define NxtOshcycle(M,t,dt) PnxtOshcycle( Mf(M), Nf(M), (t), (dt) )
#define PrvOshcycle(M,t,dt) PprvOshcycle( Mf(M), Nf(M), (t), (dt) )


/* The following macros allow us to access the different timing params
 * by name.  ('TP' stands for timing parameter.)  Valid values for S are
 * S, H, C, and P.  Note that these operations won't work as desired
 * if implemented as functions, and not macros.
 */

#define TP(M,S)                   Tptodt( M, (M->S) )
#define TPedge(M,i,S)             SHedge( M, (i), (M->S) )
#define TPcycle(M,t,S)            SHcycle( M, (t), TP(M,S) )

   /* These macros find the appropriate cycle corresponding to time t if
    * t is a timing parameter edge S.
    */

#define _NxtOTPcycle(M,t,S)       NxtOshcycle( M, (t), TP((M->other),S) )
#define _PrvOTPcycle(M,t,S)       PrvOshcycle( M, (t), TP((M->other),S) )

   /* NxtOTPcycle returns j when asked to find the next Sn edge j of the
    * other system at or after the Sm edge i of the original system (M).
    * PrvOTPcycle is similar but returns the edge at or _before_.
    */

#define NxtOTPcycle(M,i,Sm,Sn)    _NxtOTPcycle( M, TPedge(M,i,Sm), Sn )
#define PrvOTPcycle(M,i,Sm,Sn)    _PrvOTPcycle( M, TPedge(M,i,Sm), Sn )



/* 
 * Global Variables
 *
 */

systype M, N;
long fM, fN;
long Sm, Hm, Cm, Pm;
long Sn, Hn, Cn, Pn;
int quit = 0;

unsigned char ROMimage[65536];



/*
 * Basic Output Routines
 *
 */

void change_outfile( char *filename )
{
      fclose( outfile );
      outfile = fopen( filename, "wt" );
}


void show( char *fmt, ... )
{
   va_list args;

   va_start( args, fmt );
   vprintf( fmt, args );
   if ( (outmode == TEXT) || (outmode == ROM) ) {
      vfprintf( outfile, fmt, args ); 
   }
   va_end( args );
}

void fshow( char *fmt, ... )
{
   va_list args;

   va_start( args, fmt );
   vfprintf( outfile, fmt, args ); 
   va_end( args );
}

void show2( char *fmt, ... )
{
   va_list args;

   va_start( args, fmt );
   vprintf( fmt, args );
   vfprintf( outfile, fmt, args ); 
   va_end( args );
}



/*
 * Initialization Routines
 *
 */

void ClearROM()
{
   long i;

   for ( i=0; i < 65536; i++ )
      ROMimage[i] = 0xFF;
   for ( i=0; i < (1 << 12); i++ )
      ROMimage[i] = 0x55;
}

/*
 * ClearSched clears the schedule tables used for
 *   transmitting from M to N.
 */

void ClearSched( systype *M, systype *N )
{
   int i;

   for (i=0; i < MAXMN; i++) {
      M->TE[i] = 0;    M->Tsel[i] = -1;   M->TD[i] = -1;
      N->RE[i] = 0;    N->Rsel[i] = -1;   N->RQ[i] = -1;
      M->checked[i] = 0;
      N->checked[i] = 0;
   }
}

/*
 * initcnt fills the cnt array with the proper counter values
 */

void initcnt( systype *M )
{
   int i;

   for (i=0; i < MAXMN; i++) {
      M->cnt[i] = i % (M->f);
   }
}

void initMN( systype *M, long fM, long Sm, long Hm, long Cm, long Pm,
             systype *N, long fN, long Sn, long Hn, long Cn, long Pn )
{
   int i;

   M->f = fM;
   M->other = N;

   M->S = -Sm; 
   M->H = Hm; 
   M->C = Cm; 
   M->P = Pm;

   N->f = fN;
   N->other = M;
 
   N->S = -Sn; 
   N->H = Hn; 
   N->C = Cn; 
   N->P = Pn;

   ClearSched( M, N );
   initcnt( M );
   ClearSched( N, M );
   initcnt( N );
}			     



/*
 * Scheduler Algorithms
 *
 */

/* No Flow Control.  Always use register 0. */

int UnctlTransmit( systype *M, systype *N )
{
   int i;

   for (i = 0; i < M->f; i++) {
         M->TE[i] = 1;
         M->Tsel[i] = 0;
   }
   for (i = 0; i < N->f; i++) {
         N->RE[i] = 1;
         N->Rsel[i] = 0;
   }
   return( 0 );
}

/* 
 * Original single-buffer greedy algorithm.
 *
 * This is adapted from Pratt and Ward's original algorithm, described
 * in section 3.3 of my thesis.
 *
 */

int _OrigTransmit( systype *M, systype *N, int begin )
{
   long i, j;
   long i1, j1, nexti;
   int count = 0;
   long Pi, Sj, Hj, Ci;
   long tcur, tbegin;


         /* STEP 2: Begin at Pbegin.
          */

   i = begin;

   Pi = TPedge(M,i,P);
   Ci = TPedge(M,i,C);
   tbegin = Pi + Beatperiod(M);   /* Note that we're _not_ using modulo math */
   tcur = Pi;
      if (PEEK) {
         show( "\nBEGIN: i = %ld, Ci = %ld, Pi = %ld \n", i, Ci, Pi );
      }

   do {
         /* STEP 3: Pi -> Sj.  Go to next Sj.  
          */

      j = NxtOTPcycle(M,i,P,S);       
      Sj = TPedge(N,j,S);
      tcur = Sj;
         if (PEEK) {
            Hj = TPedge(N,j,H);
            show( "j = %ld, Sj = %ld, Hj = %ld \n", j, Sj, Hj );
	 }

         /* STEP 4: Go to Hj. 
          */

      Hj = TPedge(N,j,H);
      tcur = Hj;

         /* STEP 5: Hj -> Ci'.  Go to next Ci. 
          */
   
      nexti = NxtOTPcycle(N,j,H,C);   
      Ci = TPedge(M,nexti,C);
      tcur = Ci;
         if (PEEK) {
            Pi = TPedge(M,nexti,P);
            show( "i' = %ld, Ci' = %ld, Pi' = %ld \n", nexti, Ci, Pi );
         }

         /* STEP 6: Go to next Pi and check. 
          */

      Pi = TPedge(M,nexti,P);
      tcur = Pi;

         /* STEP 8: If Pbegin is not passed, mark the appropriate 
          * schedule entries. 
          */
      if ( tcur <= tbegin )  {
         i1 = mod(i-1,M->f); j1 = mod(j-1,N->f);
   
         M->TE[i1] = 1;
         M->Tsel[i1] = 0;
         M->TD[i1] = count % 0x10;
         M->Tto[mod(i,M->f)] = mod(j,N->f);

         N->RE[j1] = 1;
         N->Rsel[j1] = 0;
         N->RQ[mod(j,N->f)] = count % 0x10;
         N->Rfrom[mod(j,N->f)] = mod(i,N->f);

         count++;

          if (PEEK) {
             show( "TE[%d] and RE[%d] set. \n", 
                     mod(i-1,M->f), mod(j-1,N->f) );
          }
      }

         /* STEPS 7 and 9: If you pass or hit Pbegin, then quit, else 
          * make i = nexti and repeat the process.
          */

      i = nexti;

   } while (tcur < tbegin);

   return( count );
} 

/* _Transmit schedules a transmission from M -> N and returns 
 * the number of successful transmissions in a coincidence cycle.
 */

int _Transmit( systype *M, systype *N )
{
   int count, begin;
   int bestcount, bestbegin;
   int best;
   int found;
   
   best = min( M->f, N->f );
   bestcount = 0;
   bestbegin = 0;
   begin     = 0;

   while ( (bestcount != best) && (begin < M->f) ) {
      ClearSched( M, N );
      count = _OrigTransmit( M, N, begin );
      if (count > bestcount) {
         bestcount = count;
         bestbegin = begin;
      } 
      begin++;
   }

   /* get lowest bestbegin. (if bestcount=best, then bestbegin
    * will also be the lowest bestbegin.)
    */

   if (bestcount != best) {
      ClearSched( M, N );
      count = _OrigTransmit( M, N, bestbegin );
   }
   return count;
}


/* 
 * Lazy Double-Buffered Algorithm
 *
 */

int _DBuffLTransmit( systype *M, systype *N )
{
   long x, y;
   long eold, e;
   long f0;
   int  R;
   long i, j, i1, j1;
   long P, S, H, C;
   long m, n;
   int  c, count = 0;
   int  slowtx;
   
   slowtx = M->f <= N->f;

   if (slowtx) {
      m = Period(N);
      n = Period(M);
   } else {
      m = Period(M);
      n = Period(N);
   }

   P = Tptodt( M, M->P );   C = Tptodt( M, M->C );
   S = Tptodt( N, N->S );   H = Tptodt( N, N->H );

   x = 1;
   if ( slowtx )
      f0 = -( P - S );  /* Remember that S here is -S of S in the paper. */
   else
      f0 = -( H - C );
   y = ndiv( f0, n ) + 1; 
   eold = mod( f0, n ) - n;
   R = 0;
      if (PEEK) {
         show( "f0: %ld, e0: %ld, x0: %ld, y0: %ld\n", f0, eold, x, y );
      }
   for (c=0; c < max(M->f,N->f); c++) {
      e = eold + m;
         if (PEEK) {
            show( "eold: %ld, e: %ld, x: %ld, y: %ld\n", eold, e, x, y );
         }
      if ( e >= 0 ) {
         if (slowtx) {
            i = y; j = x;
	 } else {
            i = x; j = y;
         }

         i1 = mod(i-1,M->f); j1 = mod(j-1,N->f);
   
         M->TE[i1] = 1;
         M->Tsel[i1] = R;
         M->TD[i1] = count % 0x10;
         M->Tto[mod(i,M->f)] = mod(j+2,N->f); 

         N->RE[j1] = 1;
         N->Rsel[j1] = R;
         N->RQ[mod(j+2,N->f)] = count % 0x10;
         N->Rfrom[mod(j+2,N->f)] = mod(i,M->f); 

         R = (~R) & 1;
         y = y + 1;
         e = e - n;

         count++;         
         
      }
      x = x + 1;
      eold = e;
   }

   return( count );
} 


/* 
 * Lazy Double-Buffered Algorithm
 *
 */

int _DBuffGTransmit( systype *M, systype *N )
{
   long x, y;
   long eold, e;
   long f0;
   int  R;
   long i, j, i1, j1;
   long P, S, H, C;
   long m, n;
   int  c, count = 0;
   int  slowtx;
   
   slowtx = M->f <= N->f;

   if (slowtx) {
      m = Period(N);
      n = Period(M);
   } else {
      m = Period(M);
      n = Period(N);
   }

   P = Tptodt( M, M->P );   C = Tptodt( M, M->C );
   S = Tptodt( N, N->S );   H = Tptodt( N, N->H );

   x = 1;
   if ( !slowtx )
      x = x - 1;            /* For greedy algo, x = x_0 - 1 = 0 */
   if ( slowtx )
      f0 = -( P - S );  
   else
      f0 = -( S - P );      /* The greedy algo transformation */
   y = ndiv( f0, n ) + 1; 
   eold = mod( f0, n ) - n;
   if (!slowtx) {           /* Adjust e0 and y0 for greedy algo */
      if ( mod(f0,n) == 0 )
         y = y - 1;
      if ( eold == -n )
         eold = 0;        /*******/
   }
   R = 0;
      if (PEEK) {
         show( "f0: %ld, e0: %ld, x0: %ld, y0: %ld\n", f0, eold, x, y );
      }
   for (c=0; c < max(M->f,N->f); c++) {
      e = eold + m;
         if (PEEK) {
            show( "eold: %ld, e: %ld, x: %ld, y: %ld\n", eold, e, x, y );
         }
      if (( slowtx && (e >= 0) ) || ( !slowtx && (e > 0) )) {
         if (slowtx) {
            i = y; j = x;
	 } else {
            i = x; j = y;
         }

         i1 = mod(i-1,M->f); j1 = mod(j-1,N->f);
   
         M->TE[i1] = 1;
         M->Tsel[i1] = R;
         M->TD[i1] = count % 0x10;
         M->Tto[mod(i,M->f)] = mod(j,N->f); 

         N->RE[j1] = 1;
         N->Rsel[j1] = R;
         N->RQ[mod(j,N->f)] = count % 0x10;
         N->Rfrom[mod(j,N->f)] = mod(i,M->f); 

         R = (~R) & 1;
         y = y + 1;
         e = e - n;

         count++;         
         
      }
      x = x + 1;
      eold = e;
   }

   return( count );
} 


/* 
 * Generalized Algorithm
 *
 */

int _GBuff2Transmit( systype *M, systype *N )
{
   long x, y;
   long eold, e, emax, emin;
   long f0;
   int  R;
   long i, j, i1, j1;
   long P, S, H, C;
   long m, n;
   int  c, count = 0;
   int  slowtx;
   
   slowtx = M->f <= N->f;
/*                         Uncomment this to test slower system's algo.
   if (slowtx)
      _slowGBuff2Transmit( M, N );
   else {
*/
   if (slowtx) {
      m = Period(N);
      n = Period(M);
   } else {
      m = Period(M);
      n = Period(N);
   }

   P = Tptodt( M, M->P );   C = Tptodt( M, M->C );
   S = Tptodt( N, N->S );   H = Tptodt( N, N->H );

   emax = n - ( (P-C) + (H-S) );
   emin = m - n + ( (P-C) + (H-S) );

   x = 1;
   if ( !slowtx )
      x = x - 1;            /* For greedy algo, x = x_0 - 1 = 0 */
   if ( slowtx )
      f0 = -( P - S );  
   else
      f0 = -( S - P );      /* The greedy algo transformation */
   y = ndiv( f0, n ) + 1; 
   eold = mod( f0, n ) - n;
   if (!slowtx) {           /* Adjust e0 and y0 for greedy algo */
      if ( mod(f0,n) == 0 )
         y = y - 1;
      if ( eold == -n )
         eold = 0;    /*********/
   }

   R = ( slowtx ? 0 : 1 );

   for (c=0; c < max(M->f,N->f); c++) {
      e = eold + m;
      if (( slowtx && (e >= 0) ) || ( !slowtx && (e > 0) )) {
         if (slowtx) {
            i = y; j = x;
	 } else {
            i = x; j = y;
         }

         i1 = mod(i-1,M->f); j1 = mod(j-1,N->f);

         if ( !slowtx )  
            if ( R == 0 )
               R = ( (e < emin) ? 1 : 0 );
            else
               R = 0;
   
         M->TE[i1] = 1;
         M->Tsel[i1] = R;
         M->TD[i1] = count % 0x10;
         M->Tto[mod(i,M->f)] = mod(j,N->f); 

         N->RE[j1] = 1;
         N->Rsel[j1] = R;
         N->RQ[mod(j,N->f)] = count % 0x10;
         N->Rfrom[mod(j,N->f)] = mod(i,M->f); 

/* Adjust count only to measure single-buffered performance.
 * Note that as a consequence of this, TD would not be incremented
 * whenever TSel is 1.
 */
         if ( R == 0 ) 
            count++;         

         if ( slowtx )  
            if ( R == 0 )
               R = ( (e > emax) ? 1 : 0 );
            else
               R = 0;
            
         y = y + 1;
         e = e - n;
      }

      x = x + 1;
      eold = e;
   }
/* }                       Uncomment this to test slower system's algo.
*/
   return( count );
} 

/*
 * Slower system's algorithm
 *
 */

int _slowGBuff2Transmit( systype *M, systype *N )
{
   long x, y;
   long eold, e, emax, emin;
   long f0;
   int  R;
   long i, j, i1, j1;
   long P, S, H, C;
   long m, n;
   int  c, count = 0;
   int  slowtx;
   long dMN;
   
/* assume M < N */

      m = Period(N);
      n = Period(M);
      dMN = n % m;

   P = Tptodt( M, M->P );   C = Tptodt( M, M->C );
   S = Tptodt( N, N->S );   H = Tptodt( N, N->H );

   emax = n - ( (P-C) + (H-S) );
   emin = m - n + ( (P-C) + (H-S) );

   f0 = -( P - S );  
   y = ndiv( f0, n ) + 1;

   eold = mod( f0, n ) - n;
   e = mod(eold,m);

   R = 0;

   for (c=0; c < M->f; c++) {
         i = y; j = x;
         i1 = mod(i-1,M->f); j1 = mod(j-1,N->f);

         M->TE[i1] = 1;
         M->Tsel[i1] = R;
         M->TD[i1] = count % 0x10;
         M->Tto[mod(i,M->f)] = mod(j,N->f); 

         if ( R == 0 ) 
            count++;         

         if ( slowtx )  
            if ( R == 0 )
               R = ( (e > emax) ? 1 : 0 );
            else
               R = 0;

         e = e - dMN;
         if ( e < 0 )
            e = e + m;

         y = y + 1;
   }

   return( count );
} 


/*
 * Higher-level Scheduling Routines
 *
 */

int Transmit( systype *M, systype *N )
{
   ClearSched( M, N );
   switch( algorithm ) {
      case NONE : return( UnctlTransmit( M, N ) );
      case ORIG : return( _Transmit( M, N ) );
      case DBUFFL : return( _DBuffLTransmit( M, N ) );
      case DBUFFG : return( _DBuffGTransmit( M, N ) );
      case GBUFF2 : return( _GBuff2Transmit( M, N ) );
      default : return( 0 );
   }
}

int MakeSched( systype *M, systype *N )
{
   int a, b, c, opt;

   opt = min( M->f, N->f );

      if (PEEK) {
         show("M -> N\n");
      }
   a = Transmit( M, N );
      if (PEEK) {
         show("\nN -> M\n");
      }
   b = Transmit( N, M );

   M->perf = ((double) a / (double) opt) * 100;
   N->perf = ((double) b / (double) opt) * 100;

   c   = min( M->perf, N->perf );
   
   return( c );
}



/*
 * Display Routines
 *
 */

void PlainSched( systype *M, int *schedptr )
{
   int i;

   for (i=0; i < M->f; i++)
      if (schedptr[i] == (-1))
         show("x");
      else
         show("%X",schedptr[i]);
   show("\n");
}


void DrawSched( systype *M, int *schedptr )
{
   int i, j;
   long Mw, Nw, MNgcd;

   Mw = Nf(M);
   Nw = Mf(M);
   MNgcd = gcd( Mw, Nw );
   Mw /= MNgcd;
   Nw /= MNgcd;  
 
   if ((Mw == 1) || (Nw == 1))
      Mw = 2*Mw;

   show("|");
   for( i = 0; i < M->f; i++ ) {
      for( j = 0; j < Mw-1; j++ ) {
         if (j == (Mw-1)/2 )
            if (schedptr[i] == (-1))
               show("x");
            else
               show("%X", schedptr[i] );
         else
            show(" ");
      }
      show("|");
   }
   show("\n");
}

#define PrintSched(M,sptr)  ( (DRAWSCHEDS) ? DrawSched(M,sptr) : \
                                             PlainSched(M,sptr)  )


void ShowTimes( systype *M, systype *N )
{
   show("\n");
   show( "Tref = %ld\n", Tref );
   show("TM:  TM = %ld, tS = %ld,  tH = %ld,  tC = %ld,  tP = %ld\n", 
	   Period(M), TP(M,S), TP(M,H), TP(M,C), TP(M,P));
   show("TN:  TN = %ld, tS = %ld,  tH = %ld,  tC = %ld,  tP = %ld\n", 
	   Period(N), TP(N,S), TP(N,H), TP(N,C), TP(N,P));
   show("\n");
}

void ShowSched( systype *M, systype *N )
{
   int i;

   show("\n");
   show("M:  f = %ld, S = %ld,  H = %ld,  C = %ld,  P = %ld\n", 
	    M->f, M->S, M->H, M->C, M->P);
   show("N:  f = %ld, S = %ld,  H = %ld,  C = %ld,  P = %ld\n", 
	    N->f, N->S, N->H, N->C, N->P);
   show("\n");

   show("M -> N\n");
   show("-------\n");
   show("N.RE   : ");
   PrintSched( N, N->RE );
   show("M.TE   : ");
   PrintSched( M, M->TE );
   show("-------\n");
   show("N.Rsel : ");
   PrintSched( N, N->Rsel );
   show("M.Tsel : ");
   PrintSched( M, M->Tsel );
   show("-------\n");
   show("N.RQ   : ");
   PrintSched( N, N->RQ );
   show("M.TD   : ");
   PrintSched( M, M->TD );
   
   show("\n");

   show("N -> M\n");
   show("-------\n");
   show("N.TE   : ");
   PrintSched( N, N->TE );
   show("M.RE   : ");
   PrintSched( M, M->RE );
   show("-------\n");
   show("N.Tsel : ");
   PrintSched( N, N->Tsel );
   show("M.Rsel : ");
   PrintSched( M, M->Rsel );
   show("-------\n");
   show("N.TD   : ");
   PrintSched( N, N->TD );
   show("M.RQ   : ");
   PrintSched( M, M->RQ );
   show("-------\n");

   show("Performance:  M->N = %d%%, N->M = %d%% \n", M->perf, N->perf );
}

void ROMsched( systype *M, systype *N )
{
   long baseaddr, offs;
   long Mi, Nj;
   int  REM, RselM, TE0M, TE1M;
   int  REN, RselN, TE0N, TE1N;

#define nneg( x ) ( ((x) < 0) ? 0 : x ) 
#define bit0( x ) ( (x) & 1 )


   baseaddr = ((Mf(M)-1) << (ROM_B*2))  + ((Mf(N)-1) << ROM_B);

   for (offs=0; offs < ROM_N; offs++) {
      Mi = (usewithchip ? (Mf(M) - offs + 1) : offs);
      Nj = (usewithchip ? (Mf(N) - offs + 1) : offs);
      Mi = mod( Mi, Mf(M) );
      Nj = mod( Nj, Mf(N) );

      TE0M = bit0( M->TE[Mi] & ~(nneg(M->Tsel[Mi])) );
      TE1M = bit0( M->TE[Mi] & nneg(M->Tsel[Mi]) );
      REM  = bit0( M->RE[Mi] );
      RselM = bit0( nneg( M->Rsel[Mi] ) );
      TE0N = bit0( N->TE[Nj] & ~(nneg(N->Tsel[Nj])) );
      TE1N = bit0( N->TE[Nj] & nneg(N->Tsel[Nj]) );
      REN  = bit0( N->RE[Nj] );
      RselN = bit0( nneg( N->Rsel[Nj] ) );

      ROMimage[baseaddr+offs] = TE0M + (TE1M << 1) 
                              + (REN << 2) + (RselN << 3)
                              + (TE0N << 4) + (TE1N << 5)  
                              + (REM << 6) + (RselM << 7);


      
   }
}

void IntelMCS( systype *M, systype *N )
{
   long baseaddr, offs;
   long chksum = 0;

   baseaddr = ((Mf(M)-1) << (ROM_B*2))  + ((Mf(N)-1) << ROM_B);

   show( ":%02X%04X00", ROM_N, baseaddr );
   chksum = ROM_N + (baseaddr >> 8) + (baseaddr & 0xFF);
   for (offs=0; offs < ROM_N; offs++) {
       show( "%02X", ROMimage[baseaddr+offs] );
       chksum += ROMimage[baseaddr+offs];
   }
   chksum = (-chksum) & 0xFF;
   show( "%02X\n", chksum );
} 

void LaTeXsched( systype *M, int *schedptr, int ypos )
{
   int i;
   char s[10];

#define entry(x,y,s) fshow("\\put(%d,%d){\\makebox(%d,2){%s}}\n", x, y, \
                           M->other->f, s )

   for (i=0; i < M->f; i++) {

      if (schedptr[i] == (-1))
         strcpy( s, "*" );
      else
         sprintf( s, "%d", schedptr[i] );

      entry( i*(M->other->f), ypos, s );
   }
}

void LaTeXvals( systype *M, systype *N )
{

#define label(y,s) fshow( "\\put(-1,%d){\\makebox(0,0)[r]{$%s$}}\n", y, s )
#define Lsched( y, sys, S ) label( (y+1), #S ); LaTeXsched( sys, sys->S, y );

   Lsched( 13, N, RE );
   Lsched( 15, N, Rsel );
   Lsched( 17, N, RQ );

   Lsched( -4, M, TE );
   Lsched( -6, M, Tsel );
   Lsched( -8, M, TD );

}

void LaTeXparams( systype *M, systype *N )
{
   double Mf = M->f;
   double Nf = N->f;

#define set( x, v )   fshow( "\\renewcommand{\\" #x "}{%g}\n", v )
#define gdttodelt( x ) ((double)(x) / (double) d )

   set( M, Mf );
   set( Mo, Mf+1 );
   set( nM, -Mf );

   set( N, Nf );
   set( No, Nf+1 );
   set( nN, -Nf );

   set( MN, Mf*Nf );

   set( CM, gdttodelt(TP(M,C)) );
   set( PM, gdttodelt(TP(M,P)) );
   set( CPM, gdttodelt( TP(M,P) - TP(M,C) ) );
   set( SN, gdttodelt( -TP(N,S) ) );
   set( NSN, Mf+gdttodelt( (TP(N,S)) ) );
   set( HN, gdttodelt( TP(N,H) ) );


}

void LaTeXdiag( systype *M, systype *N )
{
/* Note: This will fclose the current outfile. */
   change_outfile( "sched.tex" );

   fshow( "\\documentstyle{article}\n"
          "\\begin{document}\n"
          "\\setlength{\\unitlength}{.1in}\n"
          "\\newcounter{cM}\n"
          "\\newcounter{cN}\n"
          "\\input{setup}\n"         );
   LaTeXparams( M, N );
   fshow( "\\begin{picture}(%d,31)(-11,-9)\n"
          "\\input{clocks}\n",
          (M->f)*(N->f)+11        );
   LaTeXvals( M, N );
   fshow( "\\end{picture}\n"
          "\n"        );
   LaTeXparams( N, M );
   fshow( "\\begin{picture}(%d,27)(-11,-7)\n"
          "\\input{clocks}\n",
          (M->f)*(N->f)+11        );
   LaTeXvals( N, M );
   fshow( "\\end{picture}\n"
          "\\end{document}\n"        );
   fclose( outfile );

/* Use "at" here to resume writing log file. */
   outfile = fopen( outfilename, "at" );
}

void singlesched( systype *M, systype *N )
{
   MakeSched( M, N );
   ShowSched( M, N );
   if ( outmode == LDIAG )
      LaTeXdiag( M, N );
   else if ( outmode == ROM ) {
           ROMsched( M, N );
           IntelMCS( M, N );
	}
}

void tablesched( systype *M, systype *N )
{
   int oldM, oldN;
   int tx, rx, maxfreq;
   int LaTeX = ((outmode == LDIAG) || (outmode == LPLOT));

   oldM = M->f; oldN = N->f;
   maxfreq = max( oldM, oldN );

   if ( LaTeX ) {
      fshow( "{\\scriptsize\n"  );
      fshow( "\\begin{tabular}{l|" );
      for (rx=1;rx<=maxfreq;rx++)
         fshow( "r" );
      fshow( "}\nTransmit & \\multicolumn{%d}{c}{Receive Frequency} \\\\ \n",
             maxfreq );
   } else
      show("Transmit\tReceive Frequency\n");
  
   show2( "Freq." );
   for (rx=1;rx<=maxfreq;rx++) {
      if ( LaTeX )
          fshow( " & " );
      show( "\t");
      show2( "%d", rx );
   }
   if ( LaTeX )
      fshow( " \\\\ \\hline" );
   show2("\n" );
   for (tx = 1; tx <= maxfreq; tx++) {
      show2( "%d", tx );
      if ( LaTeX )
         fshow( " & " );
      show( "\t" );
      for (rx = 1; rx <= maxfreq; rx++) {
         M->f = tx; N->f =rx;
         MakeSched( M, N );
         show2("%d", M->perf );
         if (rx < maxfreq) {
            if ( LaTeX )
               fshow( " & " );
            show("\t");
	 } else {
            if ( LaTeX )
               fshow( " \\\\\n" );
            show("\n");
	 }
      }
   }
   if ( LaTeX )
      fshow( "\\end{tabular}\n}" );
   show( "\n" );
   M->f = oldM; N->f = oldN;
}

void ROMtable( systype *M, systype *N, long seg )
{
   int oldM, oldN;
   int tx, rx, maxfreq;
   long chksum;

   oldM = M->f; oldN = N->f;
   maxfreq = max( oldM, oldN );

   chksum = 2 + 2 + (seg >> 8) + (seg & 0xFF);
   chksum = (-chksum) & 0xFF;
   show( ":02000002%04X%02X\n", seg, chksum );

   for (tx = 1; tx <= maxfreq; tx++) {
      for (rx = 1; rx <= maxfreq; rx++) {
         M->f = tx; N->f =rx;
         MakeSched( M, N );
         ROMsched( M, N );
         IntelMCS( M, N );
      }
   }

   show( ":00000001FF\n" );
}

void bigROMtable( systype *M, systype *N ) 
{
   int algo;
   int oldalgo = algorithm;

   for (algo = 0; algo <= MAXALGO; algo++ ) {
      algorithm = algo;
      ROMtable( M, N, (algo << (3*ROM_B - 4)) );
   }
   algorithm = oldalgo;
}



/* 
 * User Interface (Menu) Data Structures and Routines
 *
 * This is a very simple menu system.  I'm tempted to use C++
 * here, but I'll keep it simple to make it portable.
 *
 */

#define MAXNAME  50
#define MAXITEMS 9

typedef struct procitem {
   char name[MAXNAME];      /* name of the item */
   char value[MAXNAME];     /* string to be printed beside the name */
   void (*proc)(char *);    /* procedure to execute when the item is chosen */
} procitem;

typedef struct procmenu {
   char name[MAXNAME];       /* name of menu */
   int  numitems;            /* number of items in menu */
   procitem *item[MAXITEMS]; /* array of pointers to items */
} procmenu;

typedef char valitem[MAXNAME]; /* valitem is just a string with the name */

typedef struct valmenu {
   char name[MAXNAME];      /* name of menu */
   int  *variable;          /* pointer to the variable to be changed */
   int  numitems;           /* number of items in menu */
   valitem item[MAXITEMS];  /* array of valitems (array of strings)  */
} valmenu;


/*
 * General ValMenu Routine
 * 
 */

void do_valmenu( valmenu *M )
{
   char s[10];
   int i, choice;
   int quit = 0;

   printf( "\n%s\n\n", M->name );
   for ( i=0; i < (M->numitems); i++ )
      printf( "%d) %s\n", i+1, M->item[i] );
   do {
      choice = *(M->variable);
      printf( "\nEnter your choice (1-%d) [%d]: ", M->numitems+1, 
              choice );   
      gets( s );
      sscanf( s, "%d", &choice );
   } while ((choice < 1) || (choice > (M->numitems)));
   *(M->variable) = choice;
}


/*
 * ValMenu definitions
 *
 */

valmenu PUnits = { "Units for Timing Parameters", &paramunits, 5,
                   { "dt", 
		     "delta t",
		     "1/d of clock period",
                     "Absolute units (clk div)", 
		     "Absolute units (clk mul)", 
                     "", "", "", ""
		   }
		 };

valmenu CompMode = { "Compute Mode", &compmode, 2,
                   { "Pair", 
		     "Table",
		     "",
                     "", "", "", "", "", ""
		   }
		 };

valmenu OutMode = { "Output Format", &outmode, 4,
		    { "Text",
                      "LaTeX schedule diagram",
                      "LaTeX schedule plot",
                      "ROM",
                      "", "", "", "", ""
		    }
		  };

valmenu Algo = { "Scheduling Algorithm", &algorithm, 4,
		 { "Single Buffered",
                   "Gen. Online (2 buffers)",
                   "Double Buffered (Greedy)",
                   "Double Buffered (Lazy)",
                   "", "", "", "", ""
                 }
	       };


/*
 * General ProcMenu Routine
 * 
 */

void do_procmenu( procmenu *M )
{
   char s[10];
   int i, choice;
   int quit = 0;

   do {
      printf( "\n>>> %s <<<\n\n", M->name );
      for ( i=0; i < (M->numitems); i++ ) {
         printf( "%d) %s", i+1, M->item[i]->name );
         if ( M->item[i]->value[0] != 0 )
            printf( "[%s]", M->item[i]->value );
         printf( "\n" );
      } 
      do {
         printf( "\nEnter your choice (1-%d, <cr> to go back) : ", 
                 M->numitems );   
         gets( s );
         if ( (s[0] == 0) )
            choice = (M->numitems+1);
         else
            sscanf( s, "%d", &choice );
      } while ((choice < 1) || (choice > (M->numitems+1)));
      printf("\n\n");
      if (choice != M->numitems+1) 
         (*(M->item[choice-1]->proc))( M->item[choice-1]->value );
   } while (choice != M->numitems+1);
}


/*
 * Input Routines
 *
 * These routines as the user for input.  Pressing CR keeps the variable's
 * old value.
 *
 */

char YN[2] = "ny";

void getYN( char *s, int *x ) 
{
   char temp[20];

   temp[0] = 0;
   printf( "%s (y/n)? [%c] ", s, YN[*x] );
   gets( temp );
   
   if ( temp[0] == 'y' )
      *x = 1;
   else if ( temp [0] == 'n' )
           *x = 0;
}

int askYN( char *s, int defval )
{
   int x = defval;

   getYN( s, &x );
   return( x );
}

void getstring( char *msg, char *s )
{
   char temp[40];

   printf( "%s [%s]: ", msg, s );
   gets( temp );
   if ( temp[0] != 0 )
      strcpy( s, temp );
}

void getlong( long *x )
{
   char temp[20];

   gets( temp );                 
   sscanf( temp, "%ld", x );    
}

#define getTP(x)   printf(#x " [%ld] : ", x);  \
                   getlong(&x);


/*
 * Menu item procedures
 *
 */

void do_Params( char *s )
{
   getTP(d);

   printf( "\nM, N [%ld, %ld] : ", fM, fN );
   gets( s );
   sscanf( s, "%ld, %ld", &fM, &fN );

   printf( "\nEnter the following timing parameters in units of %s:\n\n",
           PUnits.item[paramunits-1] );

   if ( (paramunits == CMULPS) || (paramunits == CDIVPS)) {
      getTP(Tref);
   }
   
   getTP(Sm); getTP(Hm); getTP(Cm); getTP(Pm);
   getTP(Sn); getTP(Hn); getTP(Cn); getTP(Pn);

/* Note: this call uses the _global_ variables M and N. */
   initMN( &M, fM, Sm, Hm, Cm, Pm, &N, fN, Sn, Hn, Cn, Pn );

      if (PEEK) {
         ShowTimes( &M, &N );
      }
   s[0] = 0;
} 

void converttodt( systype *M, systype *N, int oldunits )
{
   paramunits = oldunits;

   Sm = -TP(M,S);   Hm = TP(M,H);   Cm = TP(M,C);   Pm = TP(M,P);
   Sn = -TP(N,S);   Hn = TP(N,H);   Cn = TP(N,C);   Pn = TP(N,P);

   paramunits = DTUNITS;

   initMN( M, fM, Sm, Hm, Cm, Pm, N, fN, Sn, Hn, Cn, Pn );

      if (PEEK) {
         ShowTimes( M, N );
      }
}


void do_PUnits( char *s )
{
   int oldunits = paramunits;

   do_valmenu( &PUnits );
   if ((paramunits != oldunits)) {
      if (paramunits == DTUNITS) {
         if ( askYN( "Convert to dt units?", 0 ) ) {
            printf( "\nConverting to dt units ... " );
            converttodt( &M, &N, oldunits );
            printf( "\n\n" );
	 }
      }
   }
   s[0] = 0;
}

void do_CompMode( char *s )
{
   do_valmenu( &CompMode );
   s[0] = 0;
}

void do_OutMode( char *s )
{

   do_valmenu( &OutMode );
   s[0] = 0;
}

void do_Algo( char *s )
{
   do_valmenu( &Algo );
   s[0] = 0;
}

void do_Output( char *s )
{
   if (compmode == TABLE) {
      if ( outmode == ROM )
         bigROMtable( &M, &N );
      else
         tablesched( &M, &N );
   }
   else
      singlesched( &M, &N );
   s[0] = 0;
   fflush( outfile );
}

void do_OutOpts( char *s )
{
   char temp[41];
   strcpy( temp, outfilename );

   getstring( "Enter Log File Name", outfilename );
   if (strcmp(outfilename, temp)) {
      change_outfile( outfilename );
   }

   getYN( "Show intermediate results", &PEEK );
   getYN( "Draw text schedule diagram", &DRAWSCHEDS );

   getYN( "Use with prototype chip", &usewithchip );
   s[0] = 0;
}

/*
 * ProcItem definitions
 *
 */

procitem I_PUnits = { "Parameter Units", "", do_PUnits };
procitem I_Algo = { "Scheduling Algorithm", "", do_Algo };
procitem I_CompMode = { "Compute Mode", "", do_CompMode };
procitem I_OutMode = { "Output Format", "", do_OutMode };
procitem I_OutOpts = { "Output Options", "", do_OutOpts };

/* Hmm ... this is messy but I need to define this before defining 
 * do_Options. 
 */

procmenu OptionsMenu = { "Options Menu", 5, 
                         { &I_PUnits, &I_Algo, &I_CompMode, 
                           &I_OutMode, &I_OutOpts, NULL,
                           NULL, NULL, NULL
                         } 
                       };

void do_Options( char *s )
{
   do_procmenu( &OptionsMenu );
   s[0] = 0;
}



/*
 * ProcMenu definitions
 *
 */

procitem I_Params  = { "Enter Parameters", "", do_Params };
procitem I_Output  = { "Generate Output", "", do_Output };
procitem I_Options = { "Options", "", do_Options };


procmenu MainMenu = { "Main Menu", 3, 
                      { &I_Params, &I_Output, &I_Options,
                        NULL, NULL, NULL, NULL, NULL, NULL
                      } 
                    };



/*
 * Main Program Routines
 *
 */

void initdisplay()
{
   ClearROM();
   outfile = fopen( outfilename, "wt" );
}

void shutdown()
{
   fclose( outfile );
}


main()
{
   initdisplay();
   do {
      do_procmenu( &MainMenu );
      getYN( "Do you really want to quit", &quit );
   } while (!quit);
   shutdown();
}

